package com.cnooc.expertmanageminiapp.common.exception;

import com.cnooc.expertmanageminiapp.common.exception.enums.GlobalErrorCodeConstants;
import com.cnooc.expertmanageminiapp.common.response.Result;
import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolationException;
import lombok.extern.log4j.Log4j2;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.validation.BindException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;


/**
 * 全局异常处理类
 * @author: FuHongZhang
 */
@RestControllerAdvice
@Slf4j
public class GlobalExceptionHandler {

    
    /**
     * 处理参数校验异常
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public Result<String> handleMethodArgumentNotValidException(MethodArgumentNotValidException e, 
                                                               HttpServletRequest request) {
        log.error("参数校验失败: {}，请求URL: {}", e.getMessage(), request.getRequestURI());
        StringBuilder sb = new StringBuilder();
        e.getBindingResult().getFieldErrors().forEach(error -> 
            sb.append(error.getField()).append(": ").append(error.getDefaultMessage()).append("; ")
        );
        return Result.error("参数校验失败" + sb);
    }
    
    /**
     * 处理绑定异常
     */
    @ExceptionHandler(BindException.class)
    public Result<String> handleBindException(BindException e, HttpServletRequest request) {
        log.error("参数绑定失败: {}，请求URL: {}", e.getMessage(), request.getRequestURI());
        StringBuilder sb = new StringBuilder();
        e.getBindingResult().getFieldErrors().forEach(error -> 
            sb.append(error.getField()).append(": ").append(error.getDefaultMessage()).append("; ")
        );
        return Result.error("参数绑定失败" + sb);
    }

    /**
     * 处理参数验证异常
     */
    @ExceptionHandler(ConstraintViolationException.class)
    public Result<String> handleConstraintViolationException(ConstraintViolationException e,
                                                             HttpServletRequest request) {
        log.error("参数验证失败: {}，请求URL: {}", e.getMessage(), request.getRequestURI());
        StringBuilder sb = new StringBuilder();
        e.getConstraintViolations().forEach(violation ->
                sb.append(violation.getPropertyPath()).append(": ").append(violation.getMessage()).append("; ")
        );
        // 移除末尾的分号和空格
        String errorMsg = sb.length() > 0 ? sb.substring(0, sb.length() - 2) : "参数验证失败";
        return Result.error(errorMsg);
    }
    
    /**
     * 处理其他异常
     */
    @ExceptionHandler(Exception.class)
    public Result<String> handleException(Exception e, HttpServletRequest request) {
        log.error("系统异常: {}，请求URL: {}", e.getMessage(), request.getRequestURI(), e);
        return Result.error(HttpStatus.INTERNAL_SERVER_ERROR.value(), GlobalErrorCodeConstants.SYSTEM_ERROR.getMsg());
    }

    @ExceptionHandler(BusinessException.class)
    public Result<String> handleBusinessException(BusinessException e, HttpServletRequest request) {
        log.error("业务异常: {}，请求URL: {}", e.getMessage(), request.getRequestURI(), e);
        return Result.error(e.getErrorCode(), e.getMessage());
    }

    @ExceptionHandler(NullPointerException.class)
    public Result<String> handleNullPointerException(NullPointerException e, HttpServletRequest request) {
        log.error("空指针异常: {}，请求URL: {}", e.getMessage(), request.getRequestURI(), e);
        return Result.error(HttpStatus.INTERNAL_SERVER_ERROR.value(), GlobalErrorCodeConstants.SYSTEM_ERROR.getMsg());
    }
}
