package com.cnooc.expertmanageminiapp.system.controller;

import cn.hutool.captcha.ShearCaptcha;
import cn.hutool.core.codec.Base64;
import cn.hutool.core.util.IdUtil;
import com.cnooc.expertmanageminiapp.common.exception.enums.GlobalErrorCodeConstants;
import com.cnooc.expertmanageminiapp.common.utils.ValidUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.imageio.ImageIO;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * 图形验证码控制
 * @Author: FuHongZhang
 * @CreateTime: 2025-09-25  14:02
 */
@Slf4j
@RestController
@RequestMapping("/verify")
public class VerifyCodeController {

    // 验证码配置参数
    private static final int WIDTH = 150;
    private static final int HEIGHT = 50;
    private static final int CODE_LENGTH = 4;
    private static final int INTERFERENCE_COUNT = 3;
    private static final int EXPIRE_MINUTES = 1;

    private static final String CAPTCHA_KEY = "CAPTCHA_";

    @Autowired
    private StringRedisTemplate redisTemplate;

    /**
     * 获取图片验证码
     *
     * @return ResponseEntity 返回验证码ID和图片的base64位码
     */
    @GetMapping("/getImgCode")
    public ResponseEntity<Map<String, String>> getImgCode() throws IOException {
        // 1. 生成验证码
        ShearCaptcha captcha = new ShearCaptcha(WIDTH, HEIGHT, CODE_LENGTH, INTERFERENCE_COUNT);
        String code = captcha.getCode();
        // 2. 生成唯一验证码ID
        String captchaId = IdUtil.simpleUUID();
        // 3. 保存验证码到Redis
        redisTemplate.opsForValue().set(CAPTCHA_KEY + captchaId, code, EXPIRE_MINUTES, TimeUnit.MINUTES);
        // 4. 将验证码图片转为Base64
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        ImageIO.write(captcha.getImage(), "png", outputStream);
        String base64Image = "data:image/png;base64," + Base64.encode(outputStream.toByteArray());
        // 5. 构建响应
        Map<String, String> response = new HashMap<>();
        response.put("captchaId", captchaId);
        response.put("image", base64Image);

        return ResponseEntity.ok()
                .header(HttpHeaders.CACHE_CONTROL, "no-store, no-cache")
                .body(response);

    }

    /**
     * 验证图片验证码
     */
    @PostMapping("/verifyImgCode")
    public ResponseEntity<Map<String, Object>> verifyCaptcha(
            @RequestParam String captchaId,
            @RequestParam String inputCode) {

        Map<String, Object> result = new HashMap<>();

        // 1. 从缓存中获取验证码
        String storedCode = redisTemplate.opsForValue().get(CAPTCHA_KEY + captchaId);
        ValidUtils.isNotNull(storedCode, GlobalErrorCodeConstants.CODE_REDIS_KEY);
        // 2. 验证验证码
        ValidUtils.isTrue(storedCode.equals(inputCode), GlobalErrorCodeConstants.CAPTCHA_EXPIRED);
        // 3. 验证成功，删除Session中的验证码
        redisTemplate.delete(CAPTCHA_KEY + captchaId);

        result.put("success", true);
        result.put("message", "验证成功");
        return ResponseEntity.ok(result);
    }

}
