package com.cnooc.expertmanageminiapp.system.service.impl;

import cn.hutool.core.lang.Validator;
import cn.hutool.core.util.IdcardUtil;
import com.cnooc.expertmanageminiapp.common.constant.TokenConstants;
import com.cnooc.expertmanageminiapp.common.exception.BusinessException;
import com.cnooc.expertmanageminiapp.common.response.Result;
import com.cnooc.expertmanageminiapp.common.utils.JwtUtils;
import com.cnooc.expertmanageminiapp.common.utils.ValidUtils;
import com.cnooc.expertmanageminiapp.system.entity.vo.LoginVO;
import com.cnooc.expertmanageminiapp.system.entity.vo.VerifyCodeVO;
import com.cnooc.expertmanageminiapp.system.mapper.LoginMapper;
import com.cnooc.expertmanageminiapp.system.service.LoginService;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;


import java.time.Duration;
import java.util.UUID;

/**
 * @Author: songYuHang
 * @CreateTime: 2025-09-18  15:23
 */
@Service
@Slf4j
public class LoginServiceImpl implements LoginService {

    @Autowired
    private LoginMapper loginMapper;

    @Autowired
    private StringRedisTemplate redisTemplate;

    /** 身份证密码登录 */
    private static final int ID_NUMBER_PASSWORD = 1;
    /** 手机验证码登录 */
    private static final int PHONE_CODE = 2;

    /**
     * 登录逻辑
     * @param loginVO 登录
     * @return Result<String> token
     */
    @Override
    public Result<String> login(LoginVO loginVO) {
        log.info("开始执行登录逻辑");
        String token;
        switch (loginVO.getLoginType()) {
            case ID_NUMBER_PASSWORD:
                token = idNumberLogin(loginVO);
                break;
            case PHONE_CODE:
                token = phoneCodeLogin(loginVO);
                break;
            default:
                throw new BusinessException("请选择登录类型");
        }
        return Result.success(token);
    }

    /**
     * 获取验证码逻辑
     * @param phoneNumber 手机号
     * @return Result<String> token
     */
    @Override
    public Result<String> sendPhoneCode(String phoneNumber) {
        log.info("开始执行获取验证码逻辑");
        ValidUtils.isText(phoneNumber, "请输入手机号");
        ValidUtils.isTrue(Validator.isMobile(phoneNumber), "请输入正确的手机号");

        //待确认，验证码下发是调用第三方平台吗
        return Result.success("验证码已发送");
    }

    @Override
    public Result<String> changePass(LoginVO loginVO) {
        log.info("开始执行修改密码逻辑");
        return Result.success();
    }

    @Override
    public Result<String> verifyCode(VerifyCodeVO codeVO) {
        log.info("手机验证码验证逻辑");
        return null;
    }


    /**
     * 身份证密码登录
     * @param loginVO 登录参数
     * @return  token
     */
    private String idNumberLogin(LoginVO loginVO) {
        log.info("开始执行身份证登录逻辑");
        ValidUtils.isText(loginVO.getIdNumber(), "请输入身份证号");
        ValidUtils.isText(loginVO.getPassword(), "请输入密码");
        ValidUtils.isTrue(IdcardUtil.isValidCard(loginVO.getIdNumber()), "请输入正确的身份证号");
        //1.需要去库中查询，是否存在
        //2.存在校验密码
        //3.生成相应的uuid作为redis的key
        // // todo userid
        String uuidKey = UUID.randomUUID().toString();
        String token = JwtUtils.createToken(1,uuidKey);
        tokenSetRedis(token, uuidKey);



        //6.返回token
        return token;
    }

    /**
     * 手机验证码登录
     * @param loginVO 手机号登录参数
     * @return  token
     */
    private String phoneCodeLogin(LoginVO loginVO) {
        log.info("开始执行手机验证码登录逻辑");
        ValidUtils.isText(loginVO.getPhoneNumber(), "请输入手机号");
        ValidUtils.isText(loginVO.getPhoneCode(), "请输入验证码");
        ValidUtils.isTrue(Validator.isMobile(loginVO.getPhoneNumber()), "请输入正确格式的手机号");
        //1.根据手机号去库中查询是否存在
        //2.存在校验验证码
        //3.生成相应的uuid作为redis的key
        // todo userid
        String uuidKey = UUID.randomUUID().toString();
        String token = JwtUtils.createToken(1,uuidKey);
        tokenSetRedis(token, uuidKey);
        return token;
    }

    /**
     * 缓存存入token
     * @param token  token
     * @param uuidKey uuidKey
     */
    private void tokenSetRedis(String token, String uuidKey) {
        String redisTokenKey = TokenConstants.TOKEN_KEY_ + uuidKey;
        redisTemplate.opsForValue().set(redisTokenKey, token, Duration.ofHours(24));
    }
}
